/***************************************************************************/
// File name: 	FiveCo_CodeSample_RS485.c
// Description: It contains some fonctions which illustrate the use of the FiveCo's RS485 protocol
//				In this code sample, the length of the data are limited to 127 Bytes to make the
//				code simpler and clearer.
//
// Copyright: 	Fiveco Copyright 2002-2016
// Revision:  	1.0 	GF 	12.08.16 	first release
//
/***************************************************************************/

//---------------------------------------------------------------------------
// RS485 Fonctions type
#define	ucRS485_READ				0x00	// Byte : 000x xxxx		xxxxx = Reg size
#define	ucRS485_READ_ANSWER			0x20	// Byte : 001x xxxx		xxxxx = Reg size
#define	ucRS485_WRITE				0x40	// Byte : 010x xxxx		xxxxx = Reg size
#define ucRS485_EXT_MESSAGES		0xE0    // Byte : 111x xxxx     xxxxx = Ext message type
#define	ucRS485_REG_INDEX_ERROR		0xE0	// Byte : 1110 0000
#define	ucRS485_FRAMEID				0xE1	// Byte : 1110 0001
#define	ucRS485_FRAMEID_ANSWER		0xE2	// Byte : 1110 0010
#define	ucRS485_ENDFRAME			0xE3	// Byte : 1110 0011
#define	ucRS485_FRAME_ERROR			0xE4	// Byte : 1110 0100
#define	ucRS485_ENDFRAME_MULTI		0xE5	// Byte : 1110 0101
#define	ucRS485_ENDFRAME_MULTI_LAST	0xE6	// Byte : 1110 0110
//---------------------------------------------------------------------------
//	Add header of the packet in the buffer
//
//  pucBuffer 		: a pointer to the buffer which will contain the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  ucAddDestination	: the address of the destination of the packet
//  ucAddSource		: the address of the emitter of the packet
//  ucTransactionID	: The transaction ID of the packet (optional in the protocol)
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int initPacket(unsigned char* pucBuffer, unsigned int unBufferLength, unsigned char ucAddDestination,
				unsigned char ucAddSource, unsigned char ucTransactionID)
{
	int i=0;

	// packet minimum size is 5: 3 bytes for header (Add Dest, Add src and length) and 2 bytes for checksum
	if(unBufferLength < 5)
		return -1;

	pucBuffer[i++]=ucAddDestination;  	// the address of the destination of the packet
	pucBuffer[i++]=ucAddSource;         // the address of the emitter of the packet
	pucBuffer[i++]=0x00;				// Length of Parameters will be defined when packet is finished
	pucBuffer[i++]=ucRS485_FRAMEID;		// Add the frame ID, force the device to answer also for write only packet
	pucBuffer[i++]=ucTransactionID;

	return i;
}
//---------------------------------------------------------------------------
//	Add one register to read in the buffer
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//  ucRegAddress	: the address of the register to read
//  ucRegSize		: the size of the register to read
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int addReadRegister(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex, unsigned char ucRegAddress,
					unsigned char ucRegSize)
{
	int i=unIndex;

	// Test length of buffer, 2 bytes for the register to read + 2 bytes for the checksum
	if(unBufferLength < unIndex+4)
		return -1;

	// Test the length of the register (1-31)
	if(ucRegSize < 1 || ucRegSize > 31)
		return -1;

	// Add read function and register address to read in the bufffer
	pucBuffer[i++]=ucRS485_READ + ucRegSize;
	pucBuffer[i++]=ucRegAddress;

	return i;
}
//---------------------------------------------------------------------------
//	Add one register to write in the buffer with the value to write
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//  ucRegAddress	: the register address to write
//  ucRegSize		: the size in bytes of the register to write
//  pucRegData		: a pointer on the buffer containing the data to write. The size of this buffer must be
//			  at least equal to the register size.
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int addWriteRegister(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex, unsigned char ucRegAddress,
						unsigned char ucRegSize, unsigned char *pucRegData)
{
	int i=unIndex, j=0;

	// Test length of buffer, 2 bytes for the register to write + register size + 2 bytes for the checksum
	if(unBufferLength < unIndex+4+ucRegSize)
		return -1;

	// Test the length of the register (0-31)
	if(ucRegSize > 31)
		return -1;

	// Add register address to read in the bufffer
	pucBuffer[i++]=ucRS485_WRITE + ucRegSize;
	pucBuffer[i++]=ucRegAddress;

	// Add data to write
	while(j<ucRegSize)
		pucBuffer[i++]=pucRegData[j++];

	return i;
}
//---------------------------------------------------------------------------
//	Add one register to call in the buffer
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//  ucRegAddress	: the register address of the function to call
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int addCallFunction(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex, unsigned char ucRegAddress)
{
	int i=unIndex;

	// Test length of buffer, 2 bytes for the register to write + 2 bytes for the checksum
	if(unBufferLength < unIndex+4)
		return -1;

	// Add register address to read in the bufffer (size = 0)
	pucBuffer[i++]=ucRS485_WRITE;
	pucBuffer[i++]=ucRegAddress;

	return i;
}
//---------------------------------------------------------------------------
//	Add checksum and set length of paramters in the buffer
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int closePacket(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex)
{
	unsigned char ucCheckSum;
	unsigned int i;

	// Test length of buffer, 2 bytes for the checksum
	if(unBufferLength < unIndex+2)
		return -1;

	// Test index, limited to 127 of data including 2 bytes for checksum + 3 bytes for the header in this example
	if(unIndex > 127 - 2 + 3)
		return -1;

	// set length of paramters
	pucBuffer[2]=(unsigned char)(unIndex - 3 + 2); // header not in the length of parameters but the checksum yes;

	// Add checksum
	ucCheckSum=0;
	pucBuffer[unIndex++] = ucRS485_ENDFRAME;
	for(i=0; i<unIndex; i++)
		ucCheckSum += pucBuffer[i];

	pucBuffer[unIndex++] = ucCheckSum;

	return unIndex;
}
//---------------------------------------------------------------------------
//  Analyse received data (example for some register, not complete)
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unPacketLength	: the length of the data to read in the buffer referenced by pucBuffer
//
//	Return the number of bytes read, return 0 if packet not comprehensive, return -1 if error
//---------------------------------------------------------------------------
int readPacket(unsigned char *pucBuffer, unsigned int unPacketLength)
{
	unsigned char ucDestinationAdd, ucSourceAdd, ucDataLength;
	int i = 3;
	unsigned char ucTransactionID, aucVersion[4], ucRegisterAdd, ucMessageType, ucSize;
	unsigned int nNbPowerUp;

	// minimum packet size
	if(unPacketLength < 6)
		return 0;

	ucDestinationAdd 	= pucBuffer[0];     						// Desitnation address
	ucSourceAdd 		= pucBuffer[1];								// Source address
	ucDataLength		= pucBuffer[2];         					// Data length

	if(unPacketLength < (unsigned int)(ucDataLength + 3))
		return 0;

	while(i<ucDataLength+3)
	{
		ucMessageType = pucBuffer[i++];

		if((ucMessageType & 0xE0) == ucRS485_READ_ANSWER) 			// Read answer - 0x20 to 0x3F
		{
			ucSize = ucMessageType & 0x1F;
			ucRegisterAdd = pucBuffer[i++];

			if ((ucRegisterAdd == 0x01) && (ucSize == 4))					// Version
			{
				aucVersion[3] = pucBuffer[i++];
				aucVersion[2] = pucBuffer[i++];
				aucVersion[1] = pucBuffer[i++];
				aucVersion[0] = pucBuffer[i++];
			}
			else if ((ucRegisterAdd == 0x0B) && (ucSize == 4))				// Nb power up
			{
				nNbPowerUp = (pucBuffer[i++])    &0x000000FF;
				nNbPowerUp+= (pucBuffer[i++]<<8) &0x0000FF00;
				nNbPowerUp+= (pucBuffer[i++]<<16)&0x00FF0000;
				nNbPowerUp+= (pucBuffer[i++]<<24)&0xFF000000;
			}
			else    												// skip data of register not managed
				i =+ ucSize;
		}
		else if((ucMessageType & 0xE0) == 0xE0)						// Ext messages
		{
			if(ucMessageType == ucRS485_REG_INDEX_ERROR) 			// Register index error
				ucRegisterAdd = pucBuffer[i++];
			else if(ucMessageType == ucRS485_FRAMEID_ANSWER)		// Frame ID answer
				ucTransactionID = pucBuffer[i++];
			else if(ucMessageType == ucRS485_ENDFRAME)				// End of Frame
			{
				// Checksum verification
				i++;
			}
			else if(ucMessageType == ucRS485_FRAME_ERROR)			// Frame error
				continue;
			else if(ucMessageType == ucRS485_ENDFRAME_MULTI)		// End of Frame multiple packet
			{
				// Checksum verification
				i++;
			}
			else if(ucMessageType == ucRS485_ENDFRAME_MULTI_LAST)	// End of Frame multiple packet last packet
			{
				// Checksum verification
				i++;
			}
			else													// Unknown or not managed EXT message
			{
				// Message management
				return -1;
			}
		}
		else														// Unknown or not managed message
		{
			// Message management
			return -1;
		}
	}
	return i;
}
//---------------------------------------------------------------------------
