/***************************************************************************/
// File name: 	FiveCo_CodeSample_IP.c
// Description: It contains some fonctions which illustrate the use of the FiveCo's IP protocol
//
// Copyright: 	Fiveco Copyright 2002-2016
// Revision:  	1.0 	GF 	24.06.16 	first release
//
/***************************************************************************/

//---------------------------------------------------------------------------
//	Add header of the packet in the buffer
//
//  pucBuffer 		: a pointer to the buffer which will contain the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  isReadPacket	: Define the function ID of the packet. 0 -> write data, other values -> read data
//  usTransactionID	: The transaction ID of the packet
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int initPacket(unsigned char*pucBuffer, unsigned int unBufferLength, int isReadPacket, unsigned short usTransactionID)
{
	int i=0;
	unsigned short usFunctionID;

	// packet minimum size is 8: 6 bytes for header and 2 bytes for checksum
	if(unBufferLength < 8)
		return -1;

	if(isReadPacket)
		usFunctionID = 0x0021;
	else
		usFunctionID = 0x0022;

	pucBuffer[i++]=(usFunctionID >> 8)	&0xFF;  // Function
	pucBuffer[i++]=(usFunctionID) 		&0xFF;
	pucBuffer[i++]=(usTransactionID>>8)	&0xFF;	// Transaction ID
	pucBuffer[i++]=(usTransactionID) 	&0xFF;
	pucBuffer[i++]=0x00;						// Length of Parameters will be defined when packet is finished
	pucBuffer[i++]=0x00;

	return i;
}
//---------------------------------------------------------------------------
//	Add one register to read in the buffer
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//  ucRegAddress	: the register address to read
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int addReadRegister(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex, unsigned char ucRegAddress)
{
	int i=unIndex;

	// Test length of buffer, 1 byte for the register adress + 2 bytes for the checksum
	if(unBufferLength < unIndex+3)
		return -1;

	// test if the packet is of the right type
	if(pucBuffer[0] != 0x00 || pucBuffer[1] != 0x21)
		return -1;

	// Add register address to read in the buffer
	pucBuffer[i++]=ucRegAddress;

	return i;
}
//---------------------------------------------------------------------------
//	Add one register to write in the buffer with the value to write
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//  ucRegAddress	: the register address to write
//  ucRegSize		: the size in bytes of the register to write
//  pucRegData		: a pointer on the buffer containing the data to write. The size of this buffer must be
//			  at least equal to the register size.
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int addWriteRegister(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex, unsigned char ucRegAddress,
						unsigned char ucRegSize, unsigned char *pucRegData)
{
	int i=unIndex, j=0;

	// Test length of buffer, 1 byte for the register adress + register size + 2 bytes for the checksum
	if(unBufferLength < unIndex+ucRegSize+3)
		return -1;

	// test if the packet is of the right type
	if(pucBuffer[0] != 0x00 || pucBuffer[1] != 0x22)
		return -1;

	// Add register address to write in the bufffer
	pucBuffer[i++]=ucRegAddress;
	// Add data to write
	while(j<ucRegSize)
		pucBuffer[i++]=pucRegData[j++];

	return i;
}
//---------------------------------------------------------------------------
//	Add one register to call in the buffer
//
//  pucBuffer 		: a pointer to the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//  ucRegAddress	: the register address of the function to call
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int addCallFunction(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex, unsigned char ucRegAddress)
{
	int i=unIndex;

	// Test length of buffer, 1 byte for the register adress + 2 bytes for the checksum
	if(unBufferLength < unIndex+3)
		return -1;

	// test if the packet is of the right type
	if(pucBuffer[0] != 0x00 || pucBuffer[1] != 0x22)
		return -1;

	// Add register address to call in the bufffer
	pucBuffer[i++]=ucRegAddress;

	return i;
}
//---------------------------------------------------------------------------
//	Add the checksum calculated on the entire packet until the index position in the buffer at the index position
//	Do not use this function in your own code, but use closePacket
//
//  pucBuffer 		: a pointer on the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the acutal position in the buffer referenced by pucBuffer
//
//	No return value
//---------------------------------------------------------------------------
void addCheckSum(unsigned char* pucBuffer, unsigned int unBufferLength, unsigned int unIndex)
{
	// This function put the checksum calculated at the end of Buf
	unsigned int unSum = 0 ;
	int i, j = 0 ;
	unsigned int unTemp ;

	for(i=unIndex;i>0;i-=2)
	{
		if (i>1)
		{
			unTemp = ((pucBuffer[j]<<8)&0xFF00)+((pucBuffer[j+1])&0xFF);
			j+=2;
			unSum += (~unTemp)&0xFFFF;
		}
		else
			unSum += (~((pucBuffer[j++]<<8)&0xFF00))&0xFFFF ;
	}

	unSum = ((unSum & 0xFFFF0000)>>16) + (unSum & 0xFFFF) ;
	unSum = ((unSum & 0xFFFF0000)>>16) + (unSum & 0xFFFF) ;

	pucBuffer[j++]=(unSum & 0xFF00)>>8 ;
	pucBuffer[j]  =(unSum & 0xFF) ;
}
//---------------------------------------------------------------------------
//	Add checksum and set length of parameters in the buffer
//
//  pucBuffer 		: a pointer on the buffer which contains the data
//  unBufferLength	: the size of the buffer referenced by pucBuffer
//  unIndex		: the current position in the buffer referenced by pucBuffer
//
//	Return the current index of the buffer, return -1 if error
//---------------------------------------------------------------------------
int closePacket(unsigned char *pucBuffer, unsigned int unBufferLength, unsigned int unIndex)
{
	unsigned short usLength;

	// Test length of buffer, 2 bytes for the checksum
	if(unBufferLength < unIndex+2)
		return -1;

	// set length of paramters
	usLength = unIndex - 6; // header not in the length of parameters
	pucBuffer[4]=(usLength >> 8)&0xFF;
	pucBuffer[5]=(usLength) &0xFF;

	// Add checksum
	addCheckSum(pucBuffer, unBufferLength, unIndex);

	return unIndex+2;
}
//---------------------------------------------------------------------------
//  Analyse received data (example for some register, to be completed)
//
//  pucBuffer 		: a pointer on the buffer which contains the data
//  unPacketLength	: the length of the data to read in the buffer referenced by pucBuffer
//
//	Return the number of bytes read, return 0 if packet not comprehensive, return -1 if error
//---------------------------------------------------------------------------
int readPacket(unsigned char *pucBuffer, unsigned int unPacketLength)
{
	unsigned short usFunctionID, usTransactionID, usDataLength;

	// minimum packet size
	if(unPacketLength < 8)
		return 0;

	usFunctionID = (pucBuffer[0] << 8)&0xFF00;
	usFunctionID += pucBuffer[1];

	usTransactionID = (pucBuffer[2] << 8)&0xFF00;
	usTransactionID += pucBuffer[3];

	usDataLength = (pucBuffer[4] << 8)&0xFF00;
	usDataLength += pucBuffer[5];

	if(unPacketLength < usDataLength + 8)
		return 0;

	if(usFunctionID == 0x0024)	// Write answer
	{
		if(usDataLength != 0)
			return -1;
		else
			return 8;
	}
	else if(usFunctionID == 0x0023)	// Read answer
	{
		int i = 6;
		int nPosition, nTemp;
		float fTemperature;

		while(i<usDataLength+6)
		{
			switch (pucBuffer[i++])
			{
				case 0x26:      // Position
					nPosition = (pucBuffer[i++]<<24)&0xFF000000;
					nPosition+= (pucBuffer[i++]<<16)&0x00FF0000;
					nPosition+= (pucBuffer[i++]<<8) &0x0000FF00;
					nPosition+= (pucBuffer[i++])    &0x000000FF;
					break;
				case 0x29:      // Temperature
					nTemp = (pucBuffer[i++]<<24)&0xFF000000;
					nTemp+= (pucBuffer[i++]<<16)&0x00FF0000;
					nTemp+= (pucBuffer[i++]<<8) &0x0000FF00;
					nTemp+= (pucBuffer[i++])    &0x000000FF;
					fTemperature = ((float)nTemp)/65536;
					break;
				default:
					return -1;
			}
		}
		return i+8;
	}
	else
		return -1;
}
//---------------------------------------------------------------------------
